/*
 * This is a wrapper program to let LyX/Cygwin use either a native Win32 or
 * Cygwin version of ImageMagick's convert utility.
 *
 * This program simply changes all input pathnames into Windows or Cygwin
 * pathnames depending on which version of convert is being used.
 *
 * Warning: if ImageMagick isn't installed, the Windows' file system convert
 *          utility will be called!
 *
 * Author: Enrico Forestieri
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <unistd.h>
#include <sys/cygwin.h>
#include <sys/wait.h>

#define strfcpy(A,B,C) strncpy(A,B,C), *(A+(C)-1)=0

extern char *__progname;

int main(int ac, char **av)
{
    char *path = getenv("PATH");
    char *dir;
    char convert[3 * PATH_MAX];
    char buf[PATH_MAX];
    int fail = 1;
    int size;
    int len;
    int i;
    int j;
    cygwin_conv_path_t what;

    if (!path || !(path = strdup(path))) {
	fprintf(stderr, "%s: No PATH or out of memory", __progname);
	exit(1);
    }

    for (dir = strtok(path, ":"); dir; dir = strtok(NULL, ":")) {
	strfcpy(buf, dir, sizeof(buf));
	if (dir[0] != '.' && strncmp(dir, "/usr/local/bin", 14) != 0
		&& strlen(buf) + 10 < sizeof(buf)) {
	    strcat(buf, "/convert");
	    if (access(buf, X_OK) == 0) {
		fail = 0;
		break;
	    }
	}
    }

    free(path);
    if (fail) {
	fprintf(stderr, "%s: Cannot find a real convert binary\n", __progname);
	exit(2);
    }

    /* Assume native Win32 binaries are outside the cygwin tree */
    if (strncmp(buf, "/usr", 4) == 0 || strncmp(buf, "/bin", 4) == 0) {
	what = CCP_WIN_A_TO_POSIX | CCP_RELATIVE;
    } else {
	what = CCP_POSIX_TO_WIN_A | CCP_RELATIVE;
	setenv("TEMP", "/tmp", 0);
    }

    snprintf(convert, sizeof(convert), "\"%s\"", buf);

    for (i = 1; i < ac; ++i) {
	char *argi = av[i];
	char *colon = strchr(argi, ':');
	if (argi[0] == '-') {
	    strfcpy(buf, argi, sizeof(buf));
	} else if (argi[1] == ':' || !colon) {
	    len = strlen(argi);
	    size = argi[1] == ':' ? 6 : 4;
	    /* emf needs Win32 style paths even with the cygwin version */
	    if (len > size && strncmp(argi + len - 4, ".emf", 4) == 0)
		what = CCP_POSIX_TO_WIN_A | CCP_RELATIVE;
	    buf[0] = '"';
	    if (cygwin_conv_path(what, argi, buf + 1, sizeof(buf) - 2)) {
		fail = 1;
		break;
	    }
	    strncat(buf, "\"", sizeof(buf) - strlen(buf));
	} else {
	    len = colon - argi;
	    /* emf needs Win32 style paths even with the cygwin version */
	    if (len == 3 && strncmp(argi, "emf", 3) == 0)
		what = CCP_POSIX_TO_WIN_A | CCP_RELATIVE;
	    buf[0] = '"';
	    len += 2;
	    strfcpy(buf + 1, argi, len);
	    ++colon;
	    if (cygwin_conv_path(what, colon, buf + len, sizeof(buf) - len - 1)) {
		fail = 1;
		break;
	    }
	    strncat(buf, "\"", sizeof(buf) - strlen(buf));
	}
	for (j = 0; buf[j]; ++j) {
	    if (buf[j] == '\\')
		buf[j] = '/';
	}
	len = strlen(convert);
	size = sizeof(convert) - len;
	len += snprintf(convert + len, size, " %s", buf);
	if (len >= sizeof(convert)) {
	    fail = 1;
	    break;
	}
    }

    if (fail) {
	fprintf(stderr, "%s: Command too long\n", __progname);
	exit(3);
    }

    fail = system(convert);

    return WEXITSTATUS(fail);
}
